#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <ctype.h>

#include "path.h"
#include "time.h"
#include "mdata.h"
#include "main.h"

typedef struct tdFileList
{
	char				*m_szFileName;
	struct tdFileList	*m_poNext;
} tdFileList;

extern unsigned long	ulInflate();
extern void				vUpdateExternalCRC32(unsigned int _uiBytesCount);
extern unsigned char	g_ucCRC32TableHighByte;

char					g_bCheckCRC = 1;
char					g_bDirectDump = 0;
char					g_bOverwrite = 0;
char					g_bQuiet = 0;
int						g_iReadBufferSize = 2 * 1024;

#define C_iWriteBufferSize	32768

char					g_cInputFile; 
char					g_cOutputFile; 
char					*g_pcReadBuffer;
char					*g_pcWriteBuffer;
unsigned long			g_ulExternalCRC32;
unsigned long			g_ulBytesRead;
unsigned char			g_ucCharsToErase;

enum { COMMAND_NONE, COMMAND_EXTRACT, COMMAND_EXTRACT_WITH_PATH, COMMAND_LIST, COMMAND_TEST };

struct
{
	unsigned long	ulSignature;						/* 4 Central directory file header signature = 0x02014b50 */
	unsigned int	uiVersionMadeBy;					/* 2 Version made by */
	unsigned int	uiVersionNeeded;					/* 2 Version needed to extract (minimum) */
	unsigned int	uiFlags;							/* 2 General purpose bit flag */
	unsigned int	uiCompressionMethod;				/* 2 Compression method */
	unsigned int	uiLastModificationTime;				/* 2 File last modification time */
	unsigned int	uiLastModificationDate;				/* 2 File last modification date */
	unsigned long	ulCRC32;							/* 4 CRC-32 */
	unsigned long	ulCompressedSize;					/* 4 Compressed size */
	unsigned long	ulUncompressedSize;					/* 4 Uncompressed size */
	unsigned int	uiFileNameLength;					/* 2 File name length (n) */
	unsigned int	uiExtraFieldLength;					/* 2 Extra field length (m) */
	unsigned int	uiFileCommentLength;				/* 2 File comment length (k) */
	unsigned int	uiDiskNumber;						/* 2 Disk number where file starts */
	unsigned int	uiInternalFileAttributes;			/* 2 Internal file attributes */
	unsigned long	ulExternalFileAttributes;			/* 4 External file attributes */
	unsigned long	ulRelativeOffset;					/* 4 Relative offset of local file header. This is the number
														 * of bytes between the start of the first disk on which the
														 * file occurs, and the start of the local file header. This
														 * allows software reading the central directory to locate the
														 * position of the file inside the .ZIP file. */
} oEntry;

struct
{
	unsigned long	ulSignature;						/* 4 Local file header signature = 0x04034b50 (read as a
														 * little-endian number) */
	unsigned int	ulVersion;							/* 2 Version needed to extract (minimum) */
	unsigned int	ulFlags;							/* 2 General purpose bit flag */
	unsigned int	uiCompressionMethod;				/* 2 Compression method */
	unsigned int	uiLastModificationTime;				/* 2 File last modification time */
	unsigned int	uiLastModificationDate;				/* 2 File last modification date */
	unsigned long	ulCRC32;							/* 4 CRC-32 */
	unsigned long	ulCompressedSize;					/* 4 Compressed size */
	unsigned long	ulUncompressedSize;					/* 4 Uncompressed size */
	unsigned int	uiFileNameLength;					/* 2 File name length (n) */
	unsigned int	uiExtraFieldLength;					/* 2 Extra field length (m) */
} oLocalFileHeader;

struct
{
	unsigned long	ulSignature;						/* 4 End of central directory signature = 0x06054b50 */
	unsigned int	uiDiskNumber;						/* 2 Number of this disk */
	unsigned int	uiCentralDirectoryDisk;				/* 2 Disk where central directory starts */
	unsigned int	uiCentralDirectoriesCountOnDisk;	/* 2 Number of central directory records on this disk */
	unsigned int	uiCentralDirectoriesTotalCount;		/* 2 Total number of central directory records */
	unsigned long	ulCentralDirectorySize;				/* 4 Size of central directory (bytes) */
	unsigned long	ulCentralDirectoryOffset;			/* 4 Offset of start of central directory, relative to start of
														 * archive */
	unsigned int	uiCommentLength;					/* 2 Comment length (n) */
} oEOCD;

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bWildcardMatch(char *pat, char *str)
{
	/*~~~~~~~~~~~~~~~~~*/
	char	*s, *p;
	bool	star = false;
	/*~~~~~~~~~~~~~~~~~*/

loopStart:
	for(s = str, p = pat; *s; ++s, ++p)
	{
		if(*p == '?')
		{
		}
		else if(*p == '*')
		{
			star = true;
			str = s, pat = p;

			do
			{
				++pat;
			} while(*pat == '*');

			if(!*pat) return true;
			goto loopStart;
		}
		else
		{
			if(toupper(*s) != *p)
			{
				if(!star) return false;
				str++;
				goto loopStart;
			}
		}
	}

	while(*p == '*') ++p;
	return(!*p);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vErasePercentage()
{
	while(g_ucCharsToErase--) my_putch('\b');
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vExit(char *_szMessage1, char *_szMessage2, int _iCode)
{
	if(_iCode) cputs("\r\nError, ");
	cputs(_szMessage1);
	if(_szMessage2)
		puts(_szMessage2);
	else
		puts("");
	exit(_iCode);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vTerminateWithError(char *_szMessage1)
{
	vExit(_szMessage1, NULL, 1);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vThrowException(unsigned int _uiAddress)
{
	/*~~~~~~~~~~~~~~~~~~~~~*/
	static char acAddress[5];
	/*~~~~~~~~~~~~~~~~~~~~~*/

	utoa(_uiAddress, acAddress, 16);
	vExit("system exception at ", acAddress, 1);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vStatusBar(unsigned long _ulV)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	unsigned int	iPercentageRead;
	static char		acPercentage[4];
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	vErasePercentage();
	iPercentageRead = oLocalFileHeader.ulCompressedSize ? _ulV / (oLocalFileHeader.ulCompressedSize / 100) : 100;
	if(iPercentageRead > 100) iPercentageRead = 100;
	utoa(iPercentageRead, acPercentage, 10);

	my_cputs(acPercentage);
	my_putch('%');

	g_ucCharsToErase = strlen(acPercentage) + 1;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vOutOfMemory()
{
	vExit("out of memory", NULL, 1);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
FILE *poOpenFile(char *_szFileName, char *_szFlags)
{
	/*~~~~~~~~~~~~~~~~*/
	static FILE *poFile;
	/*~~~~~~~~~~~~~~~~*/

	if(!g_bOverwrite && (_szFlags[0] == 'w'))
	{
		if(poFile = fopen(_szFileName, "r"))
		{
			fclose(poFile);
			vExit("file already exists: ", _szFileName, 1);
		}
	}

	poFile = fopen(_szFileName, _szFlags);
	if(!poFile) vExit((_szFlags[0] == 'r') ? "can't open for reading: " : "can't open for writing: ", _szFileName, 1);

	return poFile;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdFileList *poAddToList(tdFileList *_poList, char *_szString)
{
	/*~~~~~~~~~~~~~~~~*/
	tdFileList	*poList;
	/*~~~~~~~~~~~~~~~~*/

	poList = (tdFileList *) malloc(sizeof(tdFileList));
	poList->m_szFileName = _szString;
	poList->m_poNext = _poList;

	return poList;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPutPadded(char *_szS, char _cPadSize, char _cPadCharacter)
{
	_cPadSize -= strlen(_szS);
	while(_cPadSize-- > 0)
	{
		my_putch(_cPadCharacter);
	}

	my_cputs(_szS);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPutSize(unsigned long _ulSize)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	static char		acBuffer[5];
	char			cScale = 0;
	static char		acScales[4] = " KMG";
	unsigned long	ulRemainder;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	if(_ulSize <= 99999)
	{
		ultoan(_ulSize, acBuffer, 10, 0);
		vPutPadded(acBuffer, 5, ' ');
	}
	else
	{
		/*~~~~~~~~~*/
		char	cLen;
		/*~~~~~~~~~*/

		while(_ulSize >= 1024)
		{
			cScale++;
			ulRemainder = _ulSize & 1023;
			_ulSize >>= 10;
		}

		ultoan(_ulSize, acBuffer, 10, 0);
		cLen = strlen(acBuffer);

		if(ulRemainder && (cLen < 3))
		{
			/*~~~~~~~~~~~*/
			char	cIndex;
			/*~~~~~~~~~~~*/

			acBuffer[cLen] = '.';
			for(cIndex = cLen; cIndex < 3; cIndex++) ulRemainder *= 10;
			ultoan(ulRemainder >> 10, acBuffer + cLen + 1, 10, 3 - cLen);
		}

		vPutPadded(acBuffer, 4, ' ');
		my_putch(acScales[cScale]);
	}

	my_putch(' ');
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
char bFixFileName(char *_szFileName)
{
	/*~~~~~~~~~~~~~~~~~~~~~~*/
	char	*pcC;
	char	*pcLastDot = NULL;
	char	bResult = 0;
	/*~~~~~~~~~~~~~~~~~~~~~~*/

	for(pcC = _szFileName; *pcC; pcC++)
	{
		switch(*pcC)
		{
		case ',':
		case ' ':
		case '+':
		case '*':
		case '?':
			*pcC = '_';
			bResult = 1;
			break;

		case '[':
			*pcC = '(';
			bResult = 1;
			break;

		case ']':
			*pcC = ')';
			bResult = 1;
			break;

		case '.':
			if(pcLastDot)
			{
				*pcLastDot = '_';
				bResult = 1;
			}

			pcLastDot = pcC;
			break;

		case '/':
			*pcC = '\\';
			pcLastDot = NULL;
			break;

		case '\\':
			pcLastDot = NULL;
			break;
		}
	}

	return bResult;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vReadCallBack()
{
	g_ulBytesRead += g_iReadBufferSize;
	vStatusBar(g_ulBytesRead);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vFormattedPut(unsigned int _uiV, char _cLength, char _cHeader, char _cFooter)
{
	/*~~~~~~~~~~~~~~~~~~*/
	static char acDate[5];
	/*~~~~~~~~~~~~~~~~~~*/

	utoa(_uiV, acDate, 10);
	vPutPadded(acDate, _cLength, _cHeader);
	my_putch(_cFooter);
}

static char				*pcRWCRCBuffers;
static char				*pcCRCBuffer;
static unsigned int		uiCRCBufferSize;
static char				bLongList = 1;
static char				*szArchiveName = NULL;
static char				*szOutFileName = NULL;
static char				*szUniqueOutputFileName = NULL;
static int				iIndex;
static unsigned char	ucCommand = COMMAND_NONE;
static tdFileList		*poWildCardList = NULL;
static tdFileList		*poFileList;
static unsigned long	ulAllFilesSize = 0;
static unsigned long	ulAllFilesCompressedSize = 0;
static FILE				*poInFile = NULL, *poOutFile = NULL;

/*
 =======================================================================================================================
 =======================================================================================================================
 */

bool bGenerateOutputFileName()
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~*/
	char	bOutFileNameModified;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~*/

	if(szUniqueOutputFileName)
	{
		szOutFileName = szUniqueOutputFileName;
		bOutFileNameModified = 1;
	}
	else
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		static char *dir, *name, *ext;
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		bOutFileNameModified = bFixFileName(szOutFileName);

		vSplitPath(szOutFileName, &dir, &name, &ext);

		if(strlen(name) > 8)
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
			unsigned int	uiDuplicateCounter;
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

			uiDuplicateCounter = 1;

			poOutFile = NULL;
			do
			{
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
				static char acDuplicateTag[8];
				static char *szS;
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

				if(poOutFile)
                {
                    fclose(poOutFile);
           			poOutFile = NULL;
                }

				*acDuplicateTag = '~';
				utoa(uiDuplicateCounter++, acDuplicateTag + 1, 10);
				strcpy(name + 8 - strlen(acDuplicateTag), acDuplicateTag);
				szS = szMakePath(dir, name, ext);
				strcpy(szOutFileName, szS);
				bOutFileNameModified = 1;
				free(szS);
			} while(poOutFile = fopen(szOutFileName, "r"));
		}

		if((ucCommand == COMMAND_EXTRACT) && dir && name)
		{
			/*~~~~~~~~~~~~~~~~~~~~~~*/
			static char *dir2, *name2;
			/*~~~~~~~~~~~~~~~~~~~~~~*/

			vSplitPath(szOutFileName, &dir2, &name2, NULL);
			strcpy(szOutFileName, name2);
			bOutFileNameModified = 1;
			free(dir2);
			free(name2);
		}

		free(dir);
		free(name);
		free(ext);
	}

	return bOutFileNameModified;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vProcess(unsigned long ulOffset, char _bGetNameFromLocalFileHeader)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	static char			*pcExtraData;
	static unsigned int uiExtraLength;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	fseek(poInFile, ulOffset, SEEK_SET);

	fread(&oLocalFileHeader, 1, sizeof(oLocalFileHeader), poInFile);
	if(oLocalFileHeader.ulSignature != 0x04034b50) vExit("bad signature", NULL, 1);

	uiExtraLength = oLocalFileHeader.uiFileNameLength + oLocalFileHeader.uiExtraFieldLength;

	if(_bGetNameFromLocalFileHeader)
	{
		pcExtraData = (char *) malloc(uiExtraLength + 1);
		fread(pcExtraData, 1, uiExtraLength, poInFile);
		pcExtraData[oLocalFileHeader.uiFileNameLength] = 0;
		szOutFileName = pcExtraData;
	}
	else
	{
		fseek(poInFile, uiExtraLength, SEEK_CUR);
	}

	if(szOutFileName[strlen(szOutFileName) - 1] == '/')
	{
		if(ucCommand < COMMAND_LIST)
		{
			my_cputs("Creating ");
			my_cputs(szOutFileName);
			bFixFileName(szOutFileName);
			vCreateDirectoryForFile(szOutFileName);
		}
	}
	else
	{
		if(ucCommand < COMMAND_LIST)
		{
			switch(oLocalFileHeader.uiCompressionMethod)
			{
			case 8:		my_cputs("Inflating "); break;
			case 0:		my_cputs("Unstoring "); break;
			default:	vExit("unhandled compression method", NULL, 1);
			}

			my_cputs(szOutFileName);

			if(bGenerateOutputFileName())
			{
				my_cputs(" to ");
				my_cputs(szOutFileName);
			}

			if(!szUniqueOutputFileName)
			{
				if(ucCommand == COMMAND_EXTRACT_WITH_PATH)
				{
					vCreateDirectoryForFile(szOutFileName);
				}

				poOutFile = poOpenFile(szOutFileName, "w");
			}
		}

		if(poOutFile && (oLocalFileHeader.ulUncompressedSize > C_iWriteBufferSize))
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
			unsigned long	ulPreviousPos;
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

			ulPreviousPos = ftell(poOutFile);
			fseek(poOutFile, oLocalFileHeader.ulUncompressedSize - 1, SEEK_CUR);
			fwrite(g_pcWriteBuffer, 1, 1, poOutFile);
   			fensure(poOutFile); 
			fseek(poOutFile, ulPreviousPos, SEEK_SET);
		}

		my_putch(' ');
		switch(oLocalFileHeader.uiCompressionMethod)
		{
		case 8:
			g_cInputFile = (char) poInFile; 
			g_cOutputFile = (char) poOutFile; 
            g_ulBytesRead = 0;  

			if(((ulInflate() ^ 0xFFFFFFFF) != oLocalFileHeader.ulCRC32) && g_bCheckCRC)
				vExit("CRC mismatch", NULL, 1);
			break;

		case 0:
			{
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
				unsigned long	ulUncompressedSize = oLocalFileHeader.ulUncompressedSize;
				/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

				g_ulExternalCRC32 = 0xFFFFFFFF;

				while(ulUncompressedSize)
				{
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
					unsigned int	uiBytesToRead;
					/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

					uiBytesToRead = ulUncompressedSize < C_iWriteBufferSize ? ulUncompressedSize : C_iWriteBufferSize;
					fread(g_pcWriteBuffer, 1, uiBytesToRead, poInFile);
					if(poOutFile) fwrite(g_pcWriteBuffer, 1, uiBytesToRead, poOutFile);
					if(g_bCheckCRC) vUpdateExternalCRC32(uiBytesToRead);
					ulUncompressedSize -= uiBytesToRead;
					vStatusBar(oLocalFileHeader.ulCompressedSize - ulUncompressedSize);
				}

				if(((g_ulExternalCRC32 ^ 0xFFFFFFFF) != oLocalFileHeader.ulCRC32) && g_bCheckCRC)
					vExit("CRC mismatch", NULL, 1);
			}
			break;
		}

		vErasePercentage();
		my_cputs(g_bCheckCRC ? "CRC OK" : "OK  ");

		if(poOutFile && !szUniqueOutputFileName)
		{
			fclose(poOutFile);
            poOutFile = NULL;
			vSetFileTime
			(
				szOutFileName,
				oLocalFileHeader.uiLastModificationTime,
				oLocalFileHeader.uiLastModificationDate
			);
		}
	}

	if(_bGetNameFromLocalFileHeader)
	{
		free(pcExtraData);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
int main(int argc, char *argv[])
{
	oomcb = vOutOfMemory;
	stacksize = 512;

	for(iIndex = 1; iIndex < argc; iIndex++)
	{
		/*~~~~~~~~~*/
		char	*szP;
		/*~~~~~~~~~*/

		szP = argv[iIndex];
		while(*szP) *szP++ = toupper(*szP);
		szP = argv[iIndex];

		if(szP[0] == '/')
		{
			switch(szP[1])
			{
			case 'D':	g_bDirectDump = 1; break;
			case 'B':	g_iReadBufferSize = atoi(szP + 2); break;
			case 'C':	szUniqueOutputFileName = szP + 2; break;
			case 'F':	g_bCheckCRC = 0; break;
			case 'O':	g_bOverwrite = 1; break;
			case 'Q':	g_bQuiet = 1; break;
			case 'S':	bLongList = 0; break;
			default:	vExit("invalid option: ", szP, 1);
			}
		}
		else
		{
			if(ucCommand == COMMAND_NONE)
			{
				switch(szP[0])
				{
				case 'E':	ucCommand = COMMAND_EXTRACT; break;
				case 'L':	ucCommand = COMMAND_LIST; break;
				case 'T':	ucCommand = COMMAND_TEST; break;
				case 'X':	ucCommand = COMMAND_EXTRACT_WITH_PATH; break;
				default:	vExit("unknown command: ", szP, 1);
				}
			}
			else if(!szArchiveName)
			{
				szArchiveName = szP;
			}
			else
			{
				poWildCardList = poAddToList(poWildCardList, szP);
			}
		}
	}

	my_puts(&g_szVersion);

	if(ucCommand == COMMAND_NONE) vExit(&g_szUsage, NULL, 0);

	if(ucCommand == COMMAND_TEST)
	{
		g_bCheckCRC = 1;
		szUniqueOutputFileName = NULL;
	}

	if(bLongList && (ucCommand >= COMMAND_LIST)) my_puts(&g_szHeader);

	uiCRCBufferSize = g_bCheckCRC ? CRC_TABLE_SIZE : 0;
	g_iReadBufferSize -= uiCRCBufferSize;
	if(g_iReadBufferSize < 512) g_iReadBufferSize = 512;

	pcRWCRCBuffers = malloc(g_iReadBufferSize + C_iWriteBufferSize + uiCRCBufferSize + 256);
	g_pcReadBuffer = (char *) ((((size_t) pcRWCRCBuffers) + 255) & 0xFF00);
	g_pcWriteBuffer = g_pcReadBuffer + g_iReadBufferSize;

	if(g_bCheckCRC)
	{
		pcCRCBuffer = g_pcWriteBuffer + C_iWriteBufferSize;
		memcpy(pcCRCBuffer, &g_aucCRC32Table, CRC_TABLE_SIZE);
		g_ucCRC32TableHighByte = ((size_t) pcCRCBuffer) >> 8;
	}

	if(szUniqueOutputFileName) poOutFile = poOpenFile(szUniqueOutputFileName, "w");

	poInFile = poOpenFile(szArchiveName, "r");

	if(g_bDirectDump)
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		tdFileList	*poNewWildCardList = NULL;
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		for(poFileList = poWildCardList; poFileList; poFileList = poFileList->m_poNext)
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
			char			*pcExtraData;
			unsigned int	uiExtraLength;
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

			fseek(poInFile, atol(poFileList->m_szFileName), SEEK_SET);

			fread(&oLocalFileHeader, 1, sizeof(oLocalFileHeader), poInFile);
			if(oLocalFileHeader.ulSignature != 0x04034b50) vExit("bad signature", NULL, 1);

			uiExtraLength = oLocalFileHeader.uiFileNameLength + oLocalFileHeader.uiExtraFieldLength;

			pcExtraData = (char *) malloc(uiExtraLength + 2);
			fread(pcExtraData, 1, uiExtraLength, poInFile);
			pcExtraData[oLocalFileHeader.uiFileNameLength] = 0;

			if(pcExtraData[oLocalFileHeader.uiFileNameLength - 1] == '/')
			{
				strcat(pcExtraData, "*");
				poNewWildCardList = poAddToList(poNewWildCardList, pcExtraData);
				while(*pcExtraData) *pcExtraData++ = toupper(*pcExtraData);
			}
			else
			{
				vProcess(atol(poFileList->m_szFileName), true);
			}
		}

		if(!poNewWildCardList)
		{
			if (poInFile) fclose(poInFile);
            if (poOutFile) fclose(poOutFile);
			exit(0);
		}

		poWildCardList = poNewWildCardList;
	}

	fseek(poInFile, -((signed long) sizeof(oEOCD)), SEEK_END);
	fread(&oEOCD, 1, sizeof(oEOCD), poInFile);
	if(oEOCD.ulSignature != 0x06054B50) vExit("bad signature", NULL, 1);

	while(oEOCD.uiCentralDirectoriesCountOnDisk--)
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		char			bProcess;
		char			*pcExtraData;
		unsigned int	uiExtraLength;
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		fseek(poInFile, oEOCD.ulCentralDirectoryOffset, SEEK_SET);

		fread(&oEntry, 1, sizeof(oEntry), poInFile);
		if(oEntry.ulSignature != 0x02014b50) vExit("bad signature", NULL, 1);

		uiExtraLength = oEntry.uiExtraFieldLength + oEntry.uiFileCommentLength + oEntry.uiFileNameLength;
		pcExtraData = (char *) malloc(uiExtraLength + 1);
		fread(pcExtraData, 1, uiExtraLength, poInFile);
		oEOCD.ulCentralDirectoryOffset = ftell(poInFile);

		szOutFileName = pcExtraData;
		szOutFileName[oEntry.uiFileNameLength] = 0;

		if(poWildCardList)
		{
			bProcess = 0;

			for(poFileList = poWildCardList; poFileList; poFileList = poFileList->m_poNext)
			{
				bProcess |= bWildcardMatch(poFileList->m_szFileName, szOutFileName);
			}
		}
		else
		{
			bProcess = 1;
		}

		if(bProcess)
		{
			g_ucCharsToErase = 0;

			if(ucCommand >= COMMAND_LIST)
			{
				if(bLongList)
				{
					/*~~~~~~~~~~~~~~~*/
					char	acCRC32[9];
					/*~~~~~~~~~~~~~~~*/

					ulAllFilesSize += oEntry.ulUncompressedSize;
					ulAllFilesCompressedSize += oEntry.ulCompressedSize;

					vPutSize(oEntry.ulUncompressedSize);
					vPutSize(oEntry.ulCompressedSize);
					ultoan(oEntry.ulCRC32, acCRC32, 16, 8);
					my_cputs(acCRC32);
					my_putch(' ');
					vFormattedPut((oEntry.uiLastModificationDate >> 0) & 31, 2, ' ', '-');
					vFormattedPut((oEntry.uiLastModificationDate >> 5) & 15, 2, '0', '-');
					vFormattedPut(((oEntry.uiLastModificationDate >> 9) & 127) + 1980, 4, ' ', ' ');
					vFormattedPut((oEntry.uiLastModificationTime >> 11) & 31, 2, ' ', ':');
					vFormattedPut((oEntry.uiLastModificationTime >> 5) & 63, 2, '0', ':');
					vFormattedPut((oEntry.uiLastModificationTime << 1) & 63, 2, '0', ' ');
				}

				my_cputs(szOutFileName);
			}

			if(ucCommand != COMMAND_LIST)
			{
				vProcess(oEntry.ulRelativeOffset, false);
			}

			my_puts("");
		}

		free(pcExtraData);
	}

	if(poOutFile) fclose(poOutFile);
	if(poInFile) fclose(poInFile);

	if(bLongList && (ucCommand >= COMMAND_LIST))
	{
		my_puts("----- ----- -------- ---------- --------");
		vPutSize(ulAllFilesSize);
		vPutSize(ulAllFilesCompressedSize);
		my_puts("");
	}

	free(pcRWCRCBuffers);
	return 0;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void my_cputs(const char *str)
{
	if(!g_bQuiet) cputs(str);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void my_puts(const char *str)
{
	if(!g_bQuiet) puts(str);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void my_putch(char c)
{
	if(!g_bQuiet) putch(c);
}
