#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <math.h>

#include "AudioInput.h"
#include "BitReader.h"
#include "sofacas.h"

static char		*g_szOutputFileName = NULL;
static char		*g_szInputFileName = NULL;

static bool		g_bListAudioDevices = false;

static int		g_iAudioInputDevice = 0;			/* -a */
static double	g_fBufferLength = 0.2;				/* -b */
static bool		g_bCutOutputFile = false;			/* -c */
static int		g_iRequestedFrequency = 44100;		/* -f */
static double	g_fHeaderBlockLength = 0.2;			/* -h */
static double	g_fHeaderLength = 0.5;				/* -H */
static double	g_fMinBauds = 500.0;				/* -m */
static double	g_fMaxBauds = 4000.0;				/* -M */
static double	g_fPhaseAdjustCoeff = 0.5;			/* -p */
static double	g_fHeaderQuality = 100.0;			/* -q */
static double	g_fHeaderScanResolution = 10.0;		/* -r */
static double	g_fSilenceRatio = 0.1;				/* -s */
static double	g_fVolumeAmplification = 1.0;		/* -v */
static double	g_fHeaderScanRatio = 0.9;			/* -w */
static double	g_fHeaderEndSensitivity = 0.125;	/* -e */

unsigned char	g_acCASHeader[] = { 0x1f, 0xa6, 0xde, 0xba, 0xcc, 0x13, 0x7d, 0x74 };

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vShowRecUsage(char *_szExecutableName)
{
/*$off*/
	printf
	("\
%s rec <cas_file> [options]\n\
\n\
Records an MSX .CAS file from an audio output device or .WAV file\n\
\n\
[options] can be one or more of:\n\
    -a<i>    Specify audio input device (%d). Use -l to list\n\
    -b<f>    Audio input device buffer length in seconds (%.2fs)\n\
    -c       Cut output .CAS file in several pieces\n\
    -e<f>    Header end sensitivity (%.3f)\n\
    -f<i>    Frequency in hertzes for audio input device (%dHz)\n\
    -h<f>    Header detection block length in seconds (%.2fs)\n\
    -H<f>    Header detection length in seconds (%.2fs)\n\
    -i<file> Use .WAV file <file> as input instead of audio input device\n\
    -l       List available audio input devices and exit\n\
    -m<f>    Minimal baud rate in bauds (%.2fBd)\n\
    -M<f>    Maximal baud rate in bauds (%.2fBd)\n\
    -p<f>    Phase-adjust coefficient (%.2f)\n\
    -q<f>    Header quality (%.2f)\n\
    -r<f>    Header scan resolution (%.2f)\n\
    -s<f>    Silence ratio (%.2f)\n\
    -v<f>    Volume amplification (%.2f)\n\
    -w<f>    Header scan ratio (%.2f)\n\
    -y       Do not ask before overwriting file\n\
\n\
<i> is an integer number\n\
<f> is a floating point number\n\
<cas_file> is a target MSX .CAS file\n\
\n\
Command line options can also be specifed in a \"%s_rec.ini\" file\n",
	_szExecutableName,
	g_iAudioInputDevice,		// -a
	g_fBufferLength,			// -b
	g_fHeaderEndSensitivity,	// -e
	g_iRequestedFrequency,		// -f
	g_fHeaderBlockLength,		// -h
	g_fHeaderLength,			// -H
	g_fMinBauds,				// -m
	g_fMaxBauds,				// -M
	g_fPhaseAdjustCoeff,		// -p
	g_fHeaderQuality,			// -q
	g_fHeaderScanResolution,	// -r
	g_fSilenceRatio,			// -s
	g_fVolumeAmplification,		// -v
	g_fHeaderScanRatio,			// -w
	_szExecutableName
	);
/*$on*/
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
char *szTimeFormatted(double _fSeconds)
{
	/*~~~~~~~~~~~~~~~~~~~~~~*/
	int			iMinutes;
	int			iSeconds;
	int			iMilliSeconds;
	static char acResult[128];
	/*~~~~~~~~~~~~~~~~~~~~~~*/

	iMinutes = (int) (_fSeconds / 60.0);
	_fSeconds -= iMinutes * 60;

	iSeconds = (int) (_fSeconds);
	_fSeconds -= iSeconds;

	iMilliSeconds = (int) (_fSeconds * 10000.0);
	sprintf(acResult, "%2d:%02d.%04d", iMinutes, iSeconds, iMilliSeconds);

	return acResult;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
static void vProcessArgument(char *_szArgument)
{
	if(_szArgument[0] == '-')
	{
		switch(_szArgument[1])
		{
		case 'a':	g_iAudioInputDevice = atoi(_szArgument + 2); break;		/* -a */
		case 'b':	g_fBufferLength = atof(_szArgument + 2); break;			/* -b */
		case 'c':	g_bCutOutputFile = true; break;							/* -c */
		case 'e':	g_fHeaderEndSensitivity = atof(_szArgument + 2); break;	/* -e */
		case 'f':	g_iRequestedFrequency = atoi(_szArgument + 2); break;	/* -f */
		case 'h':	g_fHeaderBlockLength = atof(_szArgument + 2); break;	/* -h */
		case 'H':	g_fHeaderLength = atof(_szArgument + 2); break;			/* -H */
		case 'i':	g_szInputFileName = strdup(_szArgument + 2); break;		/* -i */
		case 'l':	g_bListAudioDevices = true; break;						/* -l */
		case 'm':	g_fMinBauds = atof(_szArgument + 2); break;				/* -m */
		case 'M':	g_fMaxBauds = atof(_szArgument + 2); break;				/* -M */
		case 'p':	g_fPhaseAdjustCoeff = atof(_szArgument + 2); break;		/* -p */
		case 'q':	g_fHeaderQuality = atof(_szArgument + 2); break;		/* -q */
		case 'r':	g_fHeaderScanResolution = atof(_szArgument + 2); break; /* -r */
		case 's':	g_fSilenceRatio = atof(_szArgument + 2); break;			/* -s */
		case 'v':	g_fVolumeAmplification = atof(_szArgument + 2); break;	/* -v */
		case 'w':	g_fHeaderScanRatio = atof(_szArgument + 2); break;		/* -w */
		case 'y':	g_bDoNotAskBeforeOverwritingFile = true; break;			/* -y */

		default:	vMyError(-1, "Invalid option %s.", _szArgument);
		}
	}
	else if(g_szOutputFileName == NULL)
	{
		g_szOutputFileName = strdup(_szArgument);
	}
	else
	{
		vMyError(-1, "Invalid argument %s.", _szArgument);
	}
}

typedef enum
{
	STATE_LOOKING_FOR_HEADER,
	STATE_READING_BYTE,
	STATE_EXPECTING_FIRST_START,
	STATE_EXPECTING_START,
	STATE_EXPECTING_STOP1,
	STATE_EXPECTING_STOP2,
	STATE_EOF
} tdState;

/*
 =======================================================================================================================
 =======================================================================================================================
 */
int iRecMain(int argc, char *argv[])
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	FILE				*poOutputFile;
	AudioInput			*poAudioInput;
	tdBit				eBit;
	FrequencyDetector	*poFD;
	double				fSample;
	int					iByteIndex;
	int					iIndex;
	int					iBlockSize;
	unsigned char		cByte;
	tdState				eState;
	int					iOutputFileIndex;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	vProcessArguments(argc - 1, argv + 1, "_rec", vProcessArgument);

	if(g_bListAudioDevices)
	{
		AudioInput::vListAudioDevices();
		vMyError(0, "Done!");
	}

	if(g_szOutputFileName == NULL)
	{
		vShowRecUsage(argv[0]);
		vMyError(0, "");
	}

	if(g_bCutOutputFile)
	{
		poOutputFile = NULL;
		iOutputFileIndex = 0;
	}
	else
	{
		poOutputFile = myfopen(g_szOutputFileName, true);
	}

	if(g_szInputFileName)
	{
		/*~~~~~~~~~~~~~~~~~*/
		FILE	*poInputFile;
		/*~~~~~~~~~~~~~~~~~*/

		poInputFile = myfopen(g_szInputFileName, false);
		if(poInputFile)
		{
			poAudioInput = new AudioInput(poInputFile, g_fVolumeAmplification);
		}
	}
	else
	{
		poAudioInput = new AudioInput
			(
				g_iRequestedFrequency,
				g_fVolumeAmplification,
				g_fBufferLength,
				g_iAudioInputDevice
			);

		g_bUserCanStop = true;
	}

	poAudioInput->vRecord();

	iByteIndex = 0;
	iBlockSize = (int) (poAudioInput->iFrequency() * g_fHeaderBlockLength + 0.5);
	poFD = new FrequencyDetector(poAudioInput->iFrequency(), 0, iBlockSize);

	eState = STATE_LOOKING_FOR_HEADER;

	/*~~~~~~~~~~~~~~~~~~~~~~~~~*/
	char	acHeader[17] = { 0 };
	enum { HEADER_UNDEFINED, HEADER_ASCII, HEADER_BIN, HEADER_BASIC }		eHeaderType;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~*/

	eHeaderType = HEADER_UNDEFINED;
	do
	{
		/*~~~~~~~~~~~~~~~~~~~~~*/
		double	fA, fB;
		double	fHeaderTime;
		double	fHeaderFrequency;
		double	fFrequency;
		char	acCuttedFileName[MAX_PATH];
		/*~~~~~~~~~~~~~~~~~~~~~*/

		fHeaderTime = poAudioInput->m_fTime;
		fA = 0.0;
		fB = 0.0;

		eState = STATE_EXPECTING_START;
		while((eState != STATE_EOF) && (poAudioInput->m_fTime - fHeaderTime < g_fHeaderLength))
		{
			/*~~~~~~~~~~~~~*/
			double	fFreqMin;
			double	fFreqMax;
			double	fRange;
			/*~~~~~~~~~~~~~*/

			fFreqMin = g_fMinBauds * 2.0;
			fFreqMax = g_fMaxBauds * 2.0;
			fRange = (fFreqMax - fFreqMin) * 0.5;

			fHeaderFrequency = (fFreqMax + fFreqMin) * 0.5;

			poFD->vReset();

			for(iIndex = 0; (iIndex < iBlockSize) && (eState != STATE_EOF); iIndex++)
			{
				if(!poAudioInput->bRead(&fSample))
				{
					eState = STATE_EOF;
				}
				else
				{
					poFD->vUpdate(fSample);
				}
			}

			if(eState != STATE_EOF)
			{
				/*~~~~~~~~~~~*/
				double	fPower;
				/*~~~~~~~~~~~*/

				while(fRange > 1.0)
				{
					/*~~~~~~~~~~~~~~~~~~~~~~~*/
					double	fPowerSum;
					double	fPowerFrequencySum;
					/*~~~~~~~~~~~~~~~~~~~~~~~*/

					fPowerSum = 0.0;
					fPowerFrequencySum = 0.0;
					for
					(
						fFrequency = fHeaderFrequency - fRange * 0.5;
						fFrequency < fHeaderFrequency + fRange * 0.5;
						fFrequency += fRange / g_fHeaderScanResolution
					)
					{
						poFD->vSetFrequencyToDetect(fFrequency);
						fPower = poFD->fPower(true);
						fPowerFrequencySum += fPower * fFrequency;
						fPowerSum += fPower;
					}

					fHeaderFrequency = fPowerFrequencySum / fPowerSum;
					if(fHeaderFrequency < fFreqMin)
					{
						fHeaderFrequency = fFreqMin;
					}
					else if(fHeaderFrequency > fFreqMax)
					{
						fHeaderFrequency = fFreqMax;
					}

					fRange *= g_fHeaderScanRatio;
				}

				poFD->vSetFrequencyToDetect(fFrequency);
				fPower = poFD->fPower(true);
				if(fPower > g_fHeaderQuality)
				{
					fHeaderTime += 1.0 / (double) poAudioInput->iFrequency();
					fA += fHeaderFrequency * fPower;
					fB += fPower;
				}
				else
				{
					fHeaderTime = poAudioInput->m_fTime;
					fA = 0.0;
					fB = 0.0;
				}
			}
		}

		if((eState != STATE_EOF) && (eState != STATE_LOOKING_FOR_HEADER))
		{
			/*~~~~~~~~~~~~~~~~~~~~~*/
			int			iByteCount;
			BitReader	*poBitReader;
			/*~~~~~~~~~~~~~~~~~~~~~*/

			fHeaderFrequency = fA / fB;
			printf
			(
				"[%s] Header detected (%.2fBd, quality: %.2f)\n",
				szTimeFormatted(poAudioInput->m_fTime),
				fHeaderFrequency * 0.5,
				poFD->fPower(true)
			);
			if(poOutputFile)
			{
				fwrite(g_acCASHeader, 1, sizeof(g_acCASHeader), poOutputFile);
			}

			poBitReader = new BitReader(poAudioInput, fHeaderFrequency, g_fPhaseAdjustCoeff, g_fSilenceRatio, g_fHeaderEndSensitivity);
			eState = STATE_EXPECTING_FIRST_START;

			iByteCount = 0;

			while((eState != STATE_EOF) && (eState != STATE_LOOKING_FOR_HEADER))
			{
				poBitReader->bReadNextBit(&eBit);

				if(eBit == BIT_EOF)
				{
					eState = STATE_EOF;
				}
				else if(eBit == BIT_SILENCE)
				{
					eState = STATE_LOOKING_FOR_HEADER;
					printf("[%s] Reading data (%d bytes)\n", szTimeFormatted(poAudioInput->m_fTime), iByteCount);
					printf("[%s] Silence detected\n\n", szTimeFormatted(poAudioInput->m_fTime));
					if(g_bCutOutputFile)
					{
						if(!poOutputFile)
						{
							sprintf(acCuttedFileName, "%.3d(RAW, %.2fBd) - %s", iOutputFileIndex, fHeaderFrequency * 0.5, g_szOutputFileName);
							poOutputFile = myfopen(acCuttedFileName, true);
							fwrite(g_acCASHeader, 1, sizeof(g_acCASHeader), poOutputFile);
							fwrite(acHeader, 1, iByteCount, poOutputFile);
						}
						if((eHeaderType == HEADER_UNDEFINED) && poOutputFile)
						{
							fclose(poOutputFile);
							poOutputFile = NULL;
							iOutputFileIndex++;
						}
					}
				}
				else
				{
					switch(eState)
					{
					case STATE_EXPECTING_FIRST_START:
						printf("[%s] Data found\n", szTimeFormatted(poAudioInput->m_fTime));

					case STATE_EXPECTING_START:
						if(eBit != BIT_0)
						{
							printf
							(
								"[%s] Reading data (%d bytes)\r",
								szTimeFormatted(poAudioInput->m_fTime),
								iByteCount
							);
							printf("\n[%s] Error: START mark expected\n", szTimeFormatted(poAudioInput->m_fTime));
						}
						else
						{
							iByteIndex = 8;
							cByte = 0;
							eState = STATE_READING_BYTE;
						}
						break;

					case STATE_EXPECTING_STOP1:
						if(eBit != BIT_1)
						{
							printf
							(
								"\n[%s] Error: First STOP mark expected\n",
								szTimeFormatted(poAudioInput->m_fTime)
							);
						}
						else
						{
							eState = STATE_EXPECTING_STOP2;
						}
						break;

					case STATE_EXPECTING_STOP2:
						if(eBit != BIT_1)
						{
							printf
							(
								"\n[%s] Error: Second STOP mark expected\n",
								szTimeFormatted(poAudioInput->m_fTime)
							);
						}
						else
						{
							iByteIndex = 8;
							eState = STATE_EXPECTING_START;
						}
						break;

					case STATE_READING_BYTE:
						cByte >>= 1;
						if(eBit == BIT_1) cByte |= 0x80;
						iByteIndex--;
						if(iByteIndex == 0)
						{
							if(iByteCount < 16)
							{
								acHeader[iByteCount] = cByte;
							}

							iByteCount++;
							eState = STATE_EXPECTING_STOP1;
							if(poOutputFile)
							{
								fwrite(&cByte, 1, sizeof(cByte), poOutputFile);
							}

							if(iByteCount == 16)
							{
								if(!memcmp(acHeader, ASCII, sizeof(ASCII)))
								{
									eHeaderType = HEADER_ASCII;
									printf
									(
										"[%s] ASCII descriptor \"%s\"\n",
										szTimeFormatted(poAudioInput->m_fTime),
										acHeader + 10
									);

									if(g_bCutOutputFile && (!poOutputFile))
									{
										sprintf(acCuttedFileName, "%.3d(ASCII, %.2fBd) - %s", iOutputFileIndex, fHeaderFrequency * 0.5, g_szOutputFileName);
										poOutputFile = myfopen(acCuttedFileName, true);
										fwrite(g_acCASHeader, 1, sizeof(g_acCASHeader), poOutputFile);
										fwrite(acHeader, 1, iByteCount, poOutputFile);
									}
								}
								else if(!memcmp(acHeader, BIN, sizeof(BIN)))
								{
									eHeaderType = HEADER_BIN;
									printf
									(
										"[%s] BINARY descriptor \"%s\"\n",
										szTimeFormatted(poAudioInput->m_fTime),
										acHeader + 10
									);

									if(g_bCutOutputFile && (!poOutputFile))
									{
										sprintf(acCuttedFileName, "%.3d(BINARY, %.2fBd) - %s", iOutputFileIndex, fHeaderFrequency * 0.5, g_szOutputFileName);
										poOutputFile = myfopen(acCuttedFileName, true);
										fwrite(g_acCASHeader, 1, sizeof(g_acCASHeader), poOutputFile);
										fwrite(acHeader, 1, iByteCount, poOutputFile);
									}
								}
								else if(!memcmp(acHeader, BASIC, sizeof(BASIC)))
								{
									eHeaderType = HEADER_BASIC;
									printf
									(
										"[%s] BASIC descriptor \"%s\"\n",
										szTimeFormatted(poAudioInput->m_fTime),
										acHeader + 10
									);
									if(g_bCutOutputFile && (!poOutputFile))
									{
										sprintf(acCuttedFileName, "%.3d(BASIC, %.2fBd) - %s", iOutputFileIndex, fHeaderFrequency * 0.5, g_szOutputFileName);
										poOutputFile = myfopen(acCuttedFileName, true);
										fwrite(g_acCASHeader, 1, sizeof(g_acCASHeader), poOutputFile);
										fwrite(acHeader, 1, iByteCount, poOutputFile);
									}
								}
								else
								{
									eHeaderType = HEADER_UNDEFINED;

									if(g_bCutOutputFile && (!poOutputFile))
									{
										sprintf(acCuttedFileName, "%.3d(RAW, %.2fBd) - %s", iOutputFileIndex, fHeaderFrequency * 0.5, g_szOutputFileName);
										poOutputFile = myfopen(acCuttedFileName, true);
										fwrite(g_acCASHeader, 1, sizeof(g_acCASHeader), poOutputFile);
										fwrite(acHeader, 1, iByteCount, poOutputFile);
									}
								}
							}
							else if(iByteCount == 6)
							{
								if(eHeaderType == HEADER_BIN)
								{
									printf
									(
										"[%s] BINARY info: &h%X,&h%X,&h%X\n",
										szTimeFormatted(poAudioInput->m_fTime),
										*(unsigned short int *) (acHeader + 0),
										*(unsigned short int *) (acHeader + 2),
										*(unsigned short int *) (acHeader + 4)
									);
								}
							}
							else
							{
								if((iByteCount & 15) == 0)
								{
									printf
									(
										"[%s] Reading data (%d bytes)\r",
										szTimeFormatted(poAudioInput->m_fTime),
										iByteCount
									);
								}
							}
						}
						break;
					}
				}
			}

			delete poBitReader;
		}
	} while(eState != STATE_EOF);

	printf("\n[%s] End of file\n", szTimeFormatted(poAudioInput->m_fTime));
	if(poOutputFile)
	{
		fclose(poOutputFile);
	}

	printf("\nMaximal volume: %f\n", poAudioInput->m_fMaxAmplitude);
	delete poAudioInput;

	return 0;
}
