#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <conio.h>
#include "dynarray.h"
#include "AudioOutput.h"
#include "SerialPort.h"
#include "sofacas.h"

typedef enum { WAVE_SQUARE = 0, WAVE_SINE, WAVE_SAWTOOTH, WAVE_SIZE } tdWaveForm;
typedef enum { HEADER_RAW, HEADER_ASCII, HEADER_BASIC, HEADER_BINARY } tdHeaderType;
typedef DynamicArray<unsigned char> CharArray;

bool g_bREMOn = true;

static bool g_bWaitForKey = false;
static bool g_bAddPrefixLoader = false;
static bool g_bListAudioDevices = false;
static tdWaveForm g_eWaveForm = WAVE_SQUARE;
static int g_iRequestedBaudRate = 2450;
static int g_iRequestedFrequency = 44100;
static int g_iBytesPerSample = 2;
static double g_fVolumeAmplification = 1.0;
static double g_fShortHeader = 10.0 / 6.0;
static double g_fLongHeader = 40.0 / 6.0;
static double g_fShortSilence = 1.0;
static double g_fLongSilence = 2.0;
static double g_fBufferLength = 0.2;
static char *g_szOutputFileName = NULL;
static char *g_szCOMPort = NULL;
static char *g_szInputFileName = NULL;
static int g_iAudioOutputDevice = 0;
static int g_iBlockBytes;

static char g_acBlockMessage[256];


static char *g_aszWaveFormNames[WAVE_SIZE] = { "Square", "Sine", "Sawtooth" };
static AudioOutput *g_poAudioOutput = NULL;
static int g_iRawInputSize;
static int g_iRefresh = 0;
static int g_iPlayPosition = 0;
static int g_iStartPosition = 0;
static CharArray g_oInputData;
static bool g_bEOF;

static int g_iLongPulseLength;
static int g_iShortPulseLength;

static int g_iShortHeader;
static int g_iLongHeader;
static int g_iShortSilence;
static int g_iLongSilence;

static int g_iPreviousPlayPosition = -1;

/*$off*/

static const unsigned char HEADER[8] = { 0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74 };

static const unsigned char header[32] =
{
	0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
	0xD0, 0xD0, 0x52, 0x4F, 0x4D, 0x20, 0x20, 0x20, 0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74
};

static const unsigned char inputPadding[] =
{
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char binFooter[] =
{
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char bload1[] =
{
	0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0x62, 0x6C, 0x6F, 0x61, 0x64, 0x31, 0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74,
	0x31, 0x30, 0x20, 0x42, 0x4C, 0x4F, 0x41, 0x44, 0x22, 0x63, 0x61, 0x73, 0x3A, 0x22, 0x2C, 0x52,
	0x0D, 0x0A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
} ;

static const unsigned char bload2[] =
{
	0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0x62, 0x6C, 0x6F, 0x61, 0x64, 0x32, 0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74,
	0x31, 0x30, 0x20, 0x42, 0x4C, 0x4F, 0x41, 0x44, 0x22, 0x63, 0x61, 0x73, 0x3A, 0x22, 0x2C, 0x52,
	0x0D, 0x0A, 0x32, 0x30, 0x20, 0x42, 0x4C, 0x4F, 0x41, 0x44, 0x22, 0x63, 0x61, 0x73, 0x3A, 0x22,
	0x2C, 0x52, 0x0D, 0x0A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
} ;


static unsigned char cload[] =
{
	0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0x63, 0x6C, 0x6F, 0x61, 0x64, 0x20, 0x1F, 0xA6, 0xDE, 0xBA, 0xCC, 0x13, 0x7D, 0x74,
	0x31, 0x30, 0x20, 0x50, 0x4F, 0x4B, 0x45, 0x26, 0x48, 0x46, 0x33, 0x46, 0x41, 0x2C, 0x26, 0x48,
	0x46, 0x34, 0x0D, 0x0A, 0x32, 0x30, 0x20, 0x50, 0x4F, 0x4B, 0x45, 0x26, 0x48, 0x46, 0x33, 0x46,
	0x42, 0x2C, 0x26, 0x48, 0x46, 0x42, 0x0D, 0x0A, 0x33, 0x30, 0x20, 0x50, 0x4F, 0x4B, 0x45, 0x26,
	0x48, 0x46, 0x33, 0x46, 0x38, 0x2C, 0x26, 0x48, 0x46, 0x38, 0x0D, 0x0A, 0x34, 0x30, 0x20, 0x50,
	0x4F, 0x4B, 0x45, 0x26, 0x48, 0x46, 0x33, 0x46, 0x39, 0x2C, 0x26, 0x48, 0x46, 0x42, 0x0D, 0x0A,
	0x35, 0x30, 0x20, 0x50, 0x4F, 0x4B, 0x45, 0x26, 0x48, 0x46, 0x42, 0x46, 0x34, 0x2C, 0x26, 0x48,
	0x37, 0x32, 0x0D, 0x0A, 0x36, 0x30, 0x20, 0x50, 0x4F, 0x4B, 0x45, 0x26, 0x48, 0x46, 0x42, 0x46,
	0x35, 0x2C, 0x26, 0x48, 0x37, 0x35, 0x0D, 0x0A, 0x37, 0x30, 0x20, 0x50, 0x4F, 0x4B, 0x45, 0x26,
	0x48, 0x46, 0x42, 0x46, 0x36, 0x2C, 0x26, 0x48, 0x36, 0x45, 0x0D, 0x0A, 0x38, 0x30, 0x20, 0x50,
	0x4F, 0x4B, 0x45, 0x26, 0x48, 0x46, 0x42, 0x46, 0x37, 0x2C, 0x26, 0x48, 0x44, 0x0D, 0x0A, 0x39,
	0x30, 0x20, 0x43, 0x4C, 0x4F, 0x41, 0x44, 0x0D, 0x0A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
} ;


static const unsigned char rom4000_7FFF[] =
{
	0xCD, 0x38, 0x01,			//CALL	RSLREG
	0x0F, 0x0F, 0x0F, 0x0F,		//RRCA * 4
	0xE6, 0x03,					//AND	00000011B
	0x4F,						//LD    C,A
	0x06, 0x00,					//LD    B,0
	0x21, 0xC1, 0xFC,			//LD    HL,EXPTBL
	0x09,						//ADD   HL,BC
	0x4F,						//LD    C,A
	0x7E,						//LD    A,(HL)
	0xE6, 0x80,					//AND	0x80
	0xB1,						//OR    C
	0x4F,						//LD    C,A
	0x23, 0x23, 0x23, 0x23,		//INC   HL * 4
	0x7E,						//LD    A,(HL)
	0x0F, 0x0F,					//RRCA * 2
	0xE6, 0x0C,					//AND	00001100B
	0xB1,						//OR    C
	0x26, 0x40,					//LD	H,0x40
	0xCD, 0x24, 0x00,			//CALL  ENASLT
	0x21, 0x00, 0x90,			//LD	HL,0x9000
	0x11, 0x00, 0x40,			//LD	DE,0x4000
	0x01, 0x00, 0x40,			//LD	BC,0x4000
	0xED, 0xB0,					//LDIR
	0xF3,						//DI
	0x21, 0x9A, 0xFD,			//LD	HL,HKEYI
	0x11, 0x9B, 0xFD,			//LD	DE,HKEYI + 1
	0x01, 0x2F, 0x02,			//LD	BC,HEND - HKEYI - 1
	0x36, 0xC9,					//LD	(HL),RET
	0xED, 0xB0,					//LDIR
	0xED, 0x7B, 0x4A, 0xFC,		//LD	SP,(0xFC4A)
	0x21, 0xDA, 0xFE,			//LD	HL,0xFEDA
	0xE5,						//PUSH	HL
	0x2A, 0x02, 0x40,			//LD	HL,(0x4002)
	0xE9						//JP	(HL)
};


static const unsigned char rom8000_BFFF[] =
{
	0xCD, 0x7B, 0x00,			//CALL	SETT32
	0xF3,						//DI
	0x21, 0x9A, 0xFD,			//LD	HL,HKEYI
	0x11, 0x9B, 0xFD,			//LD	DE,HKEYI + 1
	0x01, 0x2F, 0x02,			//LD	BC,HEND - HKEYI - 1
	0x36, 0xC9,					//LD	(HL),RET
	0xED, 0xB0,					//LDIR
	0xED, 0x7B, 0x4A, 0xFC,		//LD	SP,(0xFC4A)
	0x21, 0xDA, 0xFE,			//LD	HL,0xFEDA
	0xE5,						//PUSH	HL
	0x2A, 0x02, 0x80,			//LD	HL,(0x8002)
	0xE9						//JP	(HL)
};


static const unsigned char rom4000_BFFF_1[] =
{
	0xCD, 0x16, 0xD0,			//CALL	SETRAM
	0x21, 0x00, 0x90,			//LD	HL,0x9000
	0x11, 0x00, 0x40,			//LD	DE,0x4000
	0x01, 0x00, 0x40,			//LD	BC,0x4000
	0xED, 0xB0,					//LDIR
	0x3A, 0xC1, 0xFC,			//LD	A,(EXPTBL)
	0x26, 0x40,					//LD	H,0x40
	0xC3, 0x24, 0x00,			//JP	ENASLT

//SETRAM:
	0xCD, 0x38, 0x01,			//CALL	RSLREG
	0x0F, 0x0F, 0x0F, 0x0F,		//RRCA * 4
	0xE6, 0x03,					//AND	00000011B
	0x4F,						//LD    C,A
	0x06, 0x00,					//LD    B,0
	0x21, 0xC1, 0xFC,			//LD    HL,EXPTBL
	0x09,						//ADD   HL,BC
	0x4F,						//LD    C,A
	0x7E,						//LD    A,(HL)
	0xE6, 0x80,					//AND	0x80
	0xB1,						//OR    C
	0x4F,						//LD    C,A
	0x23, 0x23, 0x23, 0x23,		//INC   HL * 4
	0x7E,						//LD    A,(HL)
	0x0F, 0x0F,					//RRCA * 2
	0xE6, 0x0C,					//AND	00001100B
	0xB1,						//OR    C
	0x26, 0x40,					//LD	H,0x40
	0xC3, 0x24, 0x00			//JP    ENASLT
};


static const unsigned char rom4000_BFFF_2[] =
{
	0xCD, 0x16, 0xD0,			//CALL	SETRAM
	0xCD, 0x7B, 0x00,			//CALL	SETT32
	0xF3,						//DI
	0x21, 0x9A, 0xFD,			//LD	HL,HKEYI
	0x11, 0x9B, 0xFD,			//LD	DE,HKEYI + 1
	0x01, 0x2F, 0x02,			//LD	BC,HEND - HKEYI - 1
	0x36, 0xC9,					//LD	(HL),RET
	0xED, 0xB0,					//LDIR
	0xED, 0x7B, 0x4A, 0xFC,		//LD	SP,(0xFC4A)
	0x21, 0xDA, 0xFE,			//LD	HL,0xFEDA
	0xE5,						//PUSH	HL
	0x2A, 0x02, 0x40,			//LD	HL,(0x4002)
	0xE9						//JP	(HL)
};
/*$on*/

/*
 =======================================================================================================================
 =======================================================================================================================
 */
double fSawtooth(double f)
{
	f = fmod(f, 4.0);

	if(f < 1.0)
		f = -f;
	else
		if(f<3.0)
			f = f-2;
		else
			f=4-f;

	return f;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
double fSquare(double f)
{
	f = fmod(f, 4.0);

	if(f < 2.0)
		f = -1.0;
	else
		f = +1.0;

	return f;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
double fWaveFunction(double f)
{
	switch(g_eWaveForm)
	{
	case WAVE_SQUARE:	f = fSquare(f * 4.0); break;
	case WAVE_SINE:		f = -sin(f * 2.0 * M_PI); break;
	case WAVE_SAWTOOTH: f = fSawtooth(f * 4.0); break;
	}

	f *= g_fVolumeAmplification;

	return f;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vPrintProgressBar(bool _bForcePrint)
{
	if(_bForcePrint ||(g_iRawInputSize && ((g_iBlockBytes & 31) == 0)))
	{
		printf("[%6d] %s (%d bytes)\r", g_iPlayPosition, g_acBlockMessage, g_iBlockBytes);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWriteBit(int length)
{
	/*~~*/
	int n;
	/*~~*/

	if(g_iPlayPosition != g_iPreviousPlayPosition)
	{
		vPrintProgressBar(false);
		g_iPreviousPlayPosition = g_iPlayPosition;
	}

	for(n = 0; n < length; n++)
	{
		g_poAudioOutput->vRender(fWaveFunction(((double)n + 0.5) / (double) length));
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWriteHeader(int s)
{
	/*~~*/
	int i;
	/*~~*/

	g_iBlockBytes = 0;
	vPrintProgressBar(true);
	for(i = 0; bNoUserStop() && (i < s); i++) vWriteBit(g_iShortPulseLength);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWriteSilence(int s)
{
	/*~~*/
	int n;
	/*~~*/

	for(n = 0; bNoUserStop() && (n < s); n++)
	{
		g_poAudioOutput->vRender(0);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWriteByte(int byte)
{
	/*~~*/
	int i;
	/*~~*/

	g_iBlockBytes++;

	/* one start bit */
	vWriteBit(g_iLongPulseLength);

	/* eight data bits */
	for(i = 0; i < 8; i++)
	{
		if(byte & 1)
		{
			vWriteBit(g_iShortPulseLength);
			vWriteBit(g_iShortPulseLength);
		}
		else
			vWriteBit(g_iLongPulseLength);

		byte = byte >> 1;
	}

	/* two stop bits */
	for(i = 0; i < 4; i++) vWriteBit(g_iShortPulseLength);
}

/*
 =======================================================================================================================
    Write data until a header is detected
 =======================================================================================================================
 */
void vWriteData()
{
	bool bHeaderFound;

	g_bEOF = false;
	bHeaderFound = false;
	while((!bHeaderFound) && bNoUserStop() && (g_iPlayPosition < g_iRawInputSize))
	{
		if(!memcmp(g_oInputData.poGetData() + g_iPlayPosition, HEADER, 8))
		{
			bHeaderFound = true;
		}
		else
		{
			vWriteByte(g_oInputData[g_iPlayPosition]);
			if(g_oInputData[g_iPlayPosition] == 0x1A) g_bEOF = true;
			g_iPlayPosition++;
		}
	}

	vPrintProgressBar(true);
	printf("\n");
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool bEndsWith(const char *_szString, const char *_szSuffix)
{
	if(!_szString || !_szSuffix) return false;

	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	size_t	lenstr = strlen(_szString);
	size_t	lensuffix = strlen(_szSuffix);
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	if(lensuffix > lenstr) return false;
	return _strnicmp(_szString + lenstr - lensuffix, _szSuffix, lensuffix) == 0;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWriteBinHeader(CharArray *output, unsigned short _uiStart, unsigned int _uiEnd, unsigned int _uiExec)
{
	output->vAdd(header, sizeof(header));

	output->vAdd(_uiStart & 0xFF);
	output->vAdd((_uiStart >> 8) & 0xFF);

	output->vAdd(_uiEnd & 0xFF);
	output->vAdd((_uiEnd >> 8) & 0xFF);

	output->vAdd(_uiExec & 0xFF);
	output->vAdd((_uiExec >> 8) & 0xFF);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vRom2cas(char *_szSrc, CharArray *output)
{
	/*~~~~~~~~~~~~~~~~~~~~~~*/
	FILE			*poSrc;
	int				iSize;
	unsigned char	*acBuffer;
	/*~~~~~~~~~~~~~~~~~~~~~~*/

	poSrc = myfopen(_szSrc, false);
	fseek(poSrc, 0, SEEK_END);
	iSize = ftell(poSrc);
	acBuffer = (unsigned char *) malloc(iSize);
	fseek(poSrc, 0, SEEK_SET);
	fread(acBuffer, 1, iSize, poSrc);
	acBuffer[0] = 'B';
	fclose(poSrc);

	output->vAdd(bload2, sizeof(bload2));

	switch(acBuffer[3] & 0xC0)
	{
	case 0x40:
		if(iSize <= 16384)
		{
			/* rom4000_7FFF */
			vWriteBinHeader(output, 0x9000, 0x9000 + iSize + sizeof(rom4000_7FFF) - 1, 0x9000 + iSize);
			output->vAdd(acBuffer, iSize);
			output->vAdd(rom4000_7FFF, sizeof(rom4000_7FFF));
			output->vAdd(binFooter, sizeof(binFooter));
		}
		else
		{
			/* rom4000_BFFF */
			vWriteBinHeader(output, 0x9000, 0xCFFF + sizeof(rom4000_BFFF_1), 0xD000);
			output->vAdd(acBuffer, 16384);
			output->vAdd(rom4000_BFFF_1, sizeof(rom4000_BFFF_1));
			output->vAdd(binFooter, sizeof(binFooter));

			iSize -= 16384;

			vWriteBinHeader(output, 0x8000, 0x8000 + iSize + sizeof(rom4000_BFFF_2) - 1, 0x8000 + iSize);
			output->vAdd(acBuffer + 16384, iSize);
			output->vAdd(rom4000_BFFF_2, sizeof(rom4000_BFFF_2));
			output->vAdd(binFooter, sizeof(binFooter));
		}
		break;

	case 0x80:
		if(iSize <= 16384)
		{
			/* rom8000_BFFF */
			vWriteBinHeader(output, 0x8000, 0x8000 + iSize + sizeof(rom8000_BFFF) - 1, 0x8000 + iSize);
			output->vAdd(acBuffer, iSize);
			output->vAdd(rom8000_BFFF, sizeof(rom8000_BFFF));
			output->vAdd(binFooter, sizeof(binFooter));
		}
		else
		{
		}
		break;

	default:
		break;
	}

	free(acBuffer);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vSerialEventManager(UINT32 object, UINT32 event)
{
	/*~~~~~~~~~~~~~~~~~~~~~~*/
	SerialPort	*poSerialPort;
	/*~~~~~~~~~~~~~~~~~~~~~~*/

	poSerialPort = (SerialPort *) object;
	if(poSerialPort)
	{
		switch(event)
		{
		case SERIAL_CD_ON:
			if(!g_bREMOn)
			{
				g_bREMOn = true;
				printf("\n*** REM On  ***\n");
			}
			break;

		case SERIAL_CD_OFF:
			if(g_bREMOn)
			{
				g_bREMOn = false;
				printf("\n*** REM Off ***\n");
			}
			break;
		}
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vError(char const *, char const *, int, char const *, ...)
{
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vShowPlayUsage(char *_szExecutableName)
{
/*$off*/
	printf
	("\
%s play <cas_file> [options]\n\n\
Plays an MSX .CAS or .ROM file to an audio output device or .WAV file\n\
\n\
[options] can be one or more of:\n\
    -8       Use 8 bits per sample instead of 16\n\
    -a<i>    Specify audio output device (%d). Use -l to list\n\
    -b<f>    Audio output device buffer length in seconds (%.2fs)\n\
    -c<port> Use COM port <port> to read REM signal (e.g. COM1)\n\
    -f<i>    Frequency in hertzes (%dHz)\n\
    -g<f>    Start position in .CAS file in bytes (%d)\n\
    -h<f>    Short header duration in seconds (%.2fs)\n\
    -H<f>    Long header duration in seconds (%.2fs)\n\
    -k       Wait for [Return] key instead of long silence\n\
    -l       List available audio output devices and exit\n\
    -o<file> Output to .WAV file <file> instead of audio output device\n\
    -p       Add prefix loader to always load with RUN\"CAS:\n\
    -r<i>    Target baud rate in bauds (%dBd)\n\
    -s<f>    Short silence duration in seconds (%.2fs)\n\
    -S<f>    Long silence duration in seconds (%.2fs)\n\
    -v<f>    Volume amplification (%.2f)\n\
    -w<i>    Waveform (%s)\n\
		0: Square\n\
		1: Sine\n\
		2: Sawtooth\n\
    -y       Do not ask before overwriting file\n\
\n\
<i> is an integer number\n\
<f> is a floating point number\n\
<cas_file> is a source MSX .CAS or .ROM file\n\
\n\
Command line options can also be specifed in a \"%s_play.ini\" file\n",
	_szExecutableName,
	g_iAudioOutputDevice,
	g_fBufferLength,
	g_iRequestedFrequency,
	g_iStartPosition,
	g_fLongHeader,
	g_fShortHeader,
	g_iRequestedBaudRate,
	g_fLongSilence,
	g_fShortSilence,
	g_fVolumeAmplification,
	g_aszWaveFormNames[g_eWaveForm],
	_szExecutableName
	);
/*$on*/
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
tdHeaderType eGetHeader(long _iPosition)
{
	if(!memcmp(g_oInputData.poGetData() + _iPosition, ASCII, 10))
		return HEADER_ASCII;
	else if(!memcmp(g_oInputData.poGetData() + _iPosition, BIN, 10))
		return HEADER_BINARY;
	else if(!memcmp(g_oInputData.poGetData() + _iPosition, BASIC, 10))
		return HEADER_BASIC;
	else
		return HEADER_RAW;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
static void vProcessArgument(char *_szArgument)
{
	if(_szArgument[0] == '-')
	{
		switch(_szArgument[1])
		{
		case '8':	g_iBytesPerSample = 1; break;
		case 'a':	g_iAudioOutputDevice = atoi(_szArgument + 2); break;
		case 'b':	g_fBufferLength = atof(_szArgument + 2); break;
		case 'c':	g_szCOMPort = strdup(_szArgument + 2); break;
		case 'f':	g_iRequestedFrequency = atoi(_szArgument + 2); break;
		case 'g':	g_iStartPosition = atoi(_szArgument + 2); break;
		case 'H':	g_fLongHeader = atof(_szArgument + 2); break;
		case 'h':	g_fShortHeader = atof(_szArgument + 2); break;
		case 'k':	g_bWaitForKey = true; break;
		case 'l':	g_bListAudioDevices = true; break;
		case 'o':	g_szOutputFileName = strdup(_szArgument + 2); break;
		case 'p':	g_bAddPrefixLoader = true; break;
		case 'r':	g_iRequestedBaudRate = atoi(_szArgument + 2); break;
		case 'S':	g_fLongSilence = atof(_szArgument + 2); break;
		case 's':	g_fShortSilence = atof(_szArgument + 2); break;
		case 'v':	g_fVolumeAmplification = atof(_szArgument + 2); break;
		case 'w':	g_eWaveForm = (tdWaveForm) atoi(_szArgument + 2); break;
		case 'y':	g_bDoNotAskBeforeOverwritingFile = true; break;
		default:	vMyError(-1, "Invalid option %s.", _szArgument);
		}
	}
	else if(g_szInputFileName == NULL)
	{
		g_szInputFileName = strdup(_szArgument);
	}
	else
	{
		vMyError(-1, "Invalid argument %s.", _szArgument);
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
int iPlayMain(int argc, char *argv[])
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	char		acBlockName[7];
	SerialPort	*poSerialPort = NULL;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	vProcessArguments(argc - 1, argv + 1, "_play", vProcessArgument);

	if((g_eWaveForm >= WAVE_SIZE) || (g_eWaveForm < 0))
	{
		vMyError(-1, "Bad waveform.");
	}

	if(g_bListAudioDevices)
	{
		AudioOutput::vListAudioDevices();
		vMyError(0, "Done!");
	}

	if(g_szInputFileName == NULL)
	{
		vShowPlayUsage(argv[0]);
		vMyError(0, "");
	}

	if(bEndsWith(g_szInputFileName, ".rom"))
	{
		vRom2cas(g_szInputFileName, &g_oInputData);
		g_bWaitForKey = false;
	}
	else
	{
		/*~~~~~~~~~~~~~~~~~*/
		FILE	*poInputFile;
		/*~~~~~~~~~~~~~~~~~*/

		poInputFile = myfopen(g_szInputFileName, false);
		if(poInputFile == NULL)
		{
			vMyError(-1, "Failed opening %s.", g_szInputFileName);
		}

		fseek(poInputFile, 0, SEEK_END);
		g_oInputData.vSetSize(ftell(poInputFile));
		fseek(poInputFile, 0, SEEK_SET);
		fread(g_oInputData.poGetData(), 1, g_oInputData.iGetSize(), poInputFile);
		fclose(poInputFile);
	}

	if(g_bAddPrefixLoader)
	{
		/*~~~~~~~~~~~~~~~~~~~~~~*/
		CharArray	oNewInputData;
		/*~~~~~~~~~~~~~~~~~~~~~~*/

		switch(eGetHeader(8))
		{
		case HEADER_BINARY:
			oNewInputData.vAdd(bload1, sizeof(bload1));
			oNewInputData += g_oInputData;
			g_oInputData = oNewInputData;
			break;

		case HEADER_BASIC:
			oNewInputData.vAdd(cload, sizeof(cload));
			oNewInputData += g_oInputData;
			g_oInputData = oNewInputData;
			break;
		}
	}

	printf("Loaded %s.\n\n", g_szInputFileName);

	g_iRawInputSize = g_oInputData.iGetSize();
	g_oInputData.vAdd(inputPadding, sizeof(inputPadding));

	g_poAudioOutput = new AudioOutput(g_iRequestedFrequency, g_iBytesPerSample, g_fBufferLength, g_szOutputFileName, g_iAudioOutputDevice);

	g_iRequestedBaudRate++;
	g_iShortPulseLength = (g_poAudioOutput->iFrequency() + (g_iRequestedBaudRate << 1)) / (g_iRequestedBaudRate << 1);
	g_iLongPulseLength = g_iShortPulseLength << 1;

	printf
	(
		"\
Frequency       : %dHz\n\
Bits per sample : %d\n\
Waveform        : %s\n\
Short pulse     : %d samples\n\
Long pulse      : %d samples\n\
Baud rate       : %.2fBd\n\n",
		g_poAudioOutput->iFrequency(),
		g_iBytesPerSample * 8,
		g_aszWaveFormNames[g_eWaveForm],
		g_iShortPulseLength,
		g_iLongPulseLength,
		(double) g_poAudioOutput->iFrequency() / (double) g_iLongPulseLength
	);

	g_iShortSilence = (int) (g_poAudioOutput->iFrequency() * g_fShortSilence);
	g_iLongSilence = (int) (g_poAudioOutput->iFrequency() * g_fLongSilence);
	g_iShortHeader = (int) (g_poAudioOutput->iFrequency() * g_fShortHeader / g_iShortPulseLength);
	g_iLongHeader = (int) (g_poAudioOutput->iFrequency() * g_fLongHeader / g_iShortPulseLength);

	g_iPlayPosition = g_iStartPosition;

	if(!g_szOutputFileName)
	{
		if(g_szCOMPort)
		{
			poSerialPort = new SerialPort();
			poSerialPort->setManager(vSerialEventManager);
			poSerialPort->connect(g_szCOMPort);
		}

		g_bUserCanStop = true;
	}

	g_poAudioOutput->vPlay();

	/* Search for a header in the .cas file */
	while(bNoUserStop() && (g_iPlayPosition < g_iRawInputSize))
	{
		if(!memcmp(g_oInputData.poGetData() + g_iPlayPosition, HEADER, 8))
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/
			tdHeaderType	eHeaderType;
			/*~~~~~~~~~~~~~~~~~~~~~~~~*/

			eHeaderType = eGetHeader(g_iPlayPosition + 8);

			memcpy(acBlockName, g_oInputData.poGetData() + g_iPlayPosition + 18, 6);
			acBlockName[6] = 0;

			if(g_bWaitForKey)
			{
				g_poAudioOutput->vStop();
				printf("\nPress [ENTER] to continue...\n");
				while(bNoUserStop() && (getch() != 13)) vUpdateUserStop();
				g_poAudioOutput->vPlay();
			}


			switch(eHeaderType)
			{
			case HEADER_ASCII:
				sprintf(g_acBlockMessage, "Playing ASCII descriptor \"%s\"", acBlockName);

				vWriteSilence(g_iLongSilence);
				vWriteHeader(g_iLongHeader);
				g_iPlayPosition += 8;
				vWriteData();

				do
				{
					sprintf(g_acBlockMessage, "Playing ASCII data");
					vWriteSilence(g_iShortSilence);
					vWriteHeader(g_iShortHeader);
					g_iPlayPosition += 8;
					vWriteData();
				} while(bNoUserStop() && (!g_bEOF) && (g_iPlayPosition < g_iRawInputSize));
				break;

			case HEADER_BINARY:
				sprintf(g_acBlockMessage, "Playing BINARY descriptor \"%s\"", acBlockName);

				vWriteSilence(g_iLongSilence);
				vWriteHeader(g_iLongHeader);
				g_iPlayPosition += 8;
				vWriteData();

				sprintf(g_acBlockMessage, "Playing BINARY data (&h%X,&h%X,&h%X)",
						*(unsigned short int *) (g_oInputData.poGetData() + g_iPlayPosition + 8),
						*(unsigned short int *) (g_oInputData.poGetData() + g_iPlayPosition + 10),
						*(unsigned short int *) (g_oInputData.poGetData() + g_iPlayPosition + 12)
					);
				vWriteSilence(g_iShortSilence);
				vWriteHeader(g_iShortHeader);
				g_iPlayPosition += 8;
				vWriteData();
				break;

			case HEADER_BASIC:
				sprintf(g_acBlockMessage, "Playing BASIC descriptor \"%s\"", acBlockName);

				vWriteSilence(g_iLongSilence);
				vWriteHeader(g_iLongHeader);
				g_iPlayPosition += 8;
				vWriteData();

				sprintf(g_acBlockMessage, "Playing BASIC data");
				vWriteSilence(g_iShortSilence);
				vWriteHeader(g_iShortHeader);
				g_iPlayPosition += 8;
				vWriteData();
				break;

			case HEADER_RAW:
				sprintf(g_acBlockMessage, "Playing RAW data");

				vWriteSilence(g_iLongSilence);
				vWriteHeader(g_iLongHeader);
				g_iPlayPosition += 8;

				vWriteData();
				break;
			}
		}
		else
		{
			/* Should not occur */
			printf("\nSkipping unhandled data...\n");
			g_iPlayPosition++;
		}
		printf("\n");
	}

	vWriteSilence(g_iShortSilence);
	delete g_poAudioOutput;
	g_oInputData.vClear();

	delete poSerialPort;
	poSerialPort = NULL;

	vMyError(0, "\nDone!");

	return 0;
}
