#include <math.h>

#include "BitReader.h"
#include "FrequencyDetector.h"

/*
 =======================================================================================================================
 =======================================================================================================================
 */
BitReader::BitReader(AudioInput *_poAudioInput, double _fHeaderFrequency, double _fPhaseAdjustCoeff, double _fSilenceRatio, double _fHeaderEndSensitivity)
{
	m_poAudioInput = _poAudioInput;
	m_fPhaseAdjustCoeff = _fPhaseAdjustCoeff;
	m_fSilenceRatio = _fSilenceRatio;
	m_fHeaderEndSensitivity = _fHeaderEndSensitivity;

	m_fBaudRate = _fHeaderFrequency * 0.5;
	m_poFD1200 = new FrequencyDetector
		(
			m_poAudioInput->iFrequency(),
			_fHeaderFrequency * 0.5,
			(int) (m_poAudioInput->iFrequency() * 2.0 / _fHeaderFrequency + 0.5)
		);
	m_poFD2400 = new FrequencyDetector
		(
			m_poAudioInput->iFrequency(),
			_fHeaderFrequency,
			(int) (m_poAudioInput->iFrequency() * 2.0 / _fHeaderFrequency + 0.5)
		);

	m_fPhase = 0.0;
	m_fPeriod = 1.0 / (double) m_poAudioInput->iFrequency();
	m_fBitPeriod = 1.0 / m_fBaudRate;

	m_bCurrentBit = 1;
	m_bPreviousBit = 1;
	m_bPhaseDefined = false;

	m_fLast1200Power = 0.0;
	m_fLast2400Power = 0.0;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
BitReader::~BitReader()
{
	delete m_poFD1200;
	delete m_poFD2400;
}


/*
 =======================================================================================================================
 =======================================================================================================================
 */
bool BitReader::bReadNextBit(tdBit *_peBit)
{
	/*~~~~~~~~~~~~*/
	double	fSample;
	bool	bFound;
	/*~~~~~~~~~~~~*/

	if(!m_bPhaseDefined)
	{
		/*~~~~~~~*/
		int iIndex;
		/*~~~~~~~*/

		for(iIndex = 0; iIndex < m_poAudioInput->iFrequency() * m_fBitPeriod; iIndex++)
		{
			m_poAudioInput->bRead(&fSample);
			m_poFD1200->vUpdate(fSample);
			m_poFD2400->vUpdate(fSample);
		}
		m_fLast2400Power = m_poFD2400->fPower(false);
	}


	*_peBit = BIT_EOF;
	bFound = false;

	while(!bFound && m_poAudioInput->bRead(&fSample))
	{
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
		double	fPower1200, fPower2400;
		/*~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

		m_poFD1200->vUpdate(fSample);
		fPower1200 = m_poFD1200->fPower(false);
		m_poFD2400->vUpdate(fSample);
		fPower2400 = m_poFD2400->fPower(false);

		m_bPreviousBit = m_bCurrentBit;
		m_bCurrentBit = fPower2400 > fPower1200;

		if(!m_bPhaseDefined)
		{
			if(fPower1200 > m_fLast2400Power * m_fHeaderEndSensitivity)
			{
				m_fPhase = 0.5 * m_fBitPeriod;
				m_bPhaseDefined = true;
			}
		}
		else
		{
			if(m_bPreviousBit != m_bCurrentBit)
			{
				m_fPhase = (0.5 * m_fBitPeriod) * m_fPhaseAdjustCoeff +	m_fPhase * (1.0 - m_fPhaseAdjustCoeff);
			}

			if(m_fPhase > m_fBitPeriod)
			{
				m_fPhase -= m_fBitPeriod;
				*_peBit = BIT_SILENCE;

				if(m_bCurrentBit)
				{
					if(fPower2400 > m_fLast2400Power * m_fSilenceRatio)
					{
						m_fLast2400Power = fPower2400;
						*_peBit = BIT_1;
					}
				}
				else
				{
					if(fPower1200 > m_fLast1200Power * m_fSilenceRatio)
					{
						m_fLast1200Power = fPower1200;
						*_peBit = BIT_0;
					}
				}

				bFound = true;
			}
		}

		m_fPhase += m_fPeriod;
	}

	if(!bFound) *_peBit = BIT_EOF;

	return *_peBit != BIT_EOF;
}
