#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <stdarg.h>

typedef struct tdDMKHeader
{
	unsigned char	m_ucWriteProtected;
	unsigned char	m_ucNumberOfCylinders;
	unsigned short	m_usTrackLength;
	unsigned char	m_ucVirtualDiskOptions;
	unsigned char	m_aucUnused[11];
} tdDMKHeader;

typedef struct tdDMKTrack
{
	unsigned short	m_ausIDAMOffsets[64];
	unsigned char	*m_pucData;
	unsigned char	m_ucSectorsCount;
} tdDMKTrack;

typedef struct tdDMKSectorHeader
{
	unsigned char	m_ucFE;
	unsigned char	m_ucTrack;
	unsigned char	m_ucSide;
	unsigned char	m_ucNumber;
	unsigned char	m_ucSize;
	unsigned short	m_usCRC;
} tdDMKSectorHeader;

typedef struct tdSCPHeader
{
	unsigned char	sig[3];
	unsigned char	version;
	unsigned char	disk_type;
	unsigned char	nr_revolutions;
	unsigned char	start_track;
	unsigned char	end_track;
	unsigned char	flags;
	unsigned char	cell_width;
	unsigned short	reserved;
	unsigned int	checksum;
	unsigned int	tracks_offsets[168];
} tdSCPHeader;

typedef struct tdSCPTrackHeader
{
	unsigned char	sig[3];
	unsigned char	tracknr;
	unsigned int	duration;
	unsigned int	nr_samples;
	unsigned int	offset;
} tdSCPTrackHeader;

FILE			*g_poFile;
unsigned int	g_uiClock;
unsigned int	g_uiTotalClock;
bool			g_bPreviousBit;
unsigned int	g_uiSamplesWritten;

/*
 =======================================================================================================================
 =======================================================================================================================
 */
unsigned short usCRC16(const void *buf, size_t len, unsigned short _usCRC)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~*/
	unsigned int		i;
	const unsigned char *b = buf;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~*/

	for(i = 0; i < len; i++)
	{
		_usCRC = (unsigned char) (_usCRC >> 8) | (_usCRC << 8);
		_usCRC ^= *b++;
		_usCRC ^= (unsigned char) _usCRC >> 4;
		_usCRC ^= _usCRC << 12;
		_usCRC ^= (_usCRC & 0xff) << 5;
	}

	return _usCRC;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
unsigned short usLittleEndianToBigEndian(unsigned short _usV)
{
	return(_usV >> 8) | ((_usV << 8) & 0xFF00);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vAppendBit(bool _bBit)
{
	g_uiTotalClock += 80;
	g_uiClock += 80;

	if(_bBit)
	{
		/*~~~~~~~~~~~~~~~~~~~~~*/
		unsigned short	usSample;
		/*~~~~~~~~~~~~~~~~~~~~~*/

		usSample = usLittleEndianToBigEndian(g_uiClock);
		fwrite(&usSample, sizeof(usSample), 1, g_poFile);
		g_uiSamplesWritten++;
		g_uiClock = 0;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWriteBit(bool _bBit, bool _bWriteClock)
{
	if(_bWriteClock) vAppendBit(!(g_bPreviousBit || _bBit));
	vAppendBit(_bBit);
	if(_bWriteClock) g_bPreviousBit = _bBit;
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vWriteBits(unsigned int _ucData, int _ucSize, bool _bWriteClock)
{
	/*~~~~~~~~~~~~~~~~~~~~~~*/
	int				iBitIndex;
	unsigned int	uiMask;
	/*~~~~~~~~~~~~~~~~~~~~~~*/

	uiMask = 1 << (_ucSize - 1);

	for(iBitIndex = 0; iBitIndex < _ucSize; iBitIndex++)
	{
		vWriteBit(_ucData & uiMask, _bWriteClock);
		_ucData <<= 1;
	}
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
void vError(char *message, ...)
{
	/*~~~~~~~*/
	va_list va;
	/*~~~~~~~*/

	va_start(va, message);
	fprintf(stderr, "Error: ");
	vfprintf(stderr, message, va);
	fprintf(stderr, "\n");
	va_end(va);
	exit(-1);
}

/*
 =======================================================================================================================
 =======================================================================================================================
 */
int main(int argc, char **argv)
{
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
	tdDMKHeader			oDMKHeader;
	int					iTrackIndex;
	int					iSector;
	tdDMKTrack			*poTracks;
	int					iSectorSize;
	int					iReadCount;
	unsigned char		aucBuffer[2048];
	tdSCPHeader			oSCPHeader;
	tdSCPTrackHeader	oSCPTrackHeader;
	unsigned char ucNumberOfTracks;
	/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

	printf("\nDMK2SCP 1.0, coded by Louthrax in July 2017\n");
	printf("-------------------------------------------\n\n");

	if(argc < 3)
	{
		fprintf(stderr, "Usage: %s <source_dmk_file> <target_scp_file>\n", argv[0]);
		exit(0);
	}

	printf("Opening %s...\n", argv[1]);

	g_poFile = fopen(argv[1], "rb");
	if(!g_poFile) vError("Unable to open %s", argv[1]);

	fread(&oDMKHeader, sizeof(oDMKHeader), 1, g_poFile);

	ucNumberOfTracks = oDMKHeader.m_ucNumberOfCylinders * 2;

	printf("\n  Write protected: %s", oDMKHeader.m_ucWriteProtected ? "Yes\n" : "No\n");
	printf("  Number of tracks: %d\n", ucNumberOfTracks);
	printf("  Track length: %d bytes\n", oDMKHeader.m_usTrackLength);
	printf("  Virtual disk options: 0x%x\n\nAnalyzing DMK tracks...\n\n", oDMKHeader.m_ucVirtualDiskOptions);

	poTracks = malloc(ucNumberOfTracks * sizeof(tdDMKTrack));

	for(iTrackIndex = 0; iTrackIndex < ucNumberOfTracks; iTrackIndex++)
	{
		/*~~~~~~~~~~~~~~~~*/
		bool	bErrorFound;
		/*~~~~~~~~~~~~~~~~*/

		poTracks[iTrackIndex].m_pucData = malloc(oDMKHeader.m_usTrackLength);
		fread(poTracks[iTrackIndex].m_ausIDAMOffsets, sizeof(poTracks[iTrackIndex].m_ausIDAMOffsets), 1, g_poFile);
		fread
		(
			poTracks[iTrackIndex].m_pucData,
			oDMKHeader.m_usTrackLength - sizeof(poTracks[iTrackIndex].m_ausIDAMOffsets),
			1,
			g_poFile
		);

		printf("  Track %4d:", iTrackIndex);
		bErrorFound = false;

		for(iSector = 0; poTracks[iTrackIndex].m_ausIDAMOffsets[iSector]; iSector++)
		{
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
			unsigned short		usDataCRC;
			unsigned short		usIDAMCRC;
			unsigned short		usComputedDataCRC;
			unsigned short		usComputedIDAMCRC;
			unsigned char		*pucDataStart;
			tdDMKSectorHeader	*poSectorHeader;
			/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

			if(!(poTracks[iTrackIndex].m_ausIDAMOffsets[iSector] & 0x8000))
			{
				vError("Unhandled single density sector (sector %d in track %d)\n", iSector, iTrackIndex);
			}

			poTracks[iTrackIndex].m_ausIDAMOffsets[iSector] &= 0x3FFF;
			poTracks[iTrackIndex].m_ausIDAMOffsets[iSector] -= sizeof(poTracks[iTrackIndex].m_ausIDAMOffsets);

			poSectorHeader = (tdDMKSectorHeader *) (poTracks[iTrackIndex].m_pucData + poTracks[iTrackIndex].m_ausIDAMOffsets[iSector]);

			iSectorSize = 128 << poSectorHeader->m_ucSize;

			for
			(
				pucDataStart = poTracks[iTrackIndex].m_pucData +
					poTracks[iTrackIndex].m_ausIDAMOffsets[iSector] +
					sizeof(tdDMKSectorHeader);
				*pucDataStart != 0xA1;
				pucDataStart++
			);

			usDataCRC = usLittleEndianToBigEndian(*(unsigned short *) (pucDataStart + iSectorSize + 4));
			usIDAMCRC = usLittleEndianToBigEndian(poSectorHeader->m_usCRC);

			usComputedDataCRC = usCRC16(pucDataStart, iSectorSize + 4, 0xFFFF);
			usComputedIDAMCRC = usCRC16
				(
					poTracks[iTrackIndex].m_pucData + poTracks[iTrackIndex].m_ausIDAMOffsets[iSector] - 3,
					8,
					0xFFFF
				);

			if(iSectorSize != 512)
			{
				printf("\n    Sector %2d: unusual sector size (%d instead of 512)", iSector + 1, iSectorSize);
				bErrorFound = true;
			}

			if(iSector + 1 != poSectorHeader->m_ucNumber)
			{
				printf("\n    Sector %2d: non-matching sector number (%d)", iSector + 1, poSectorHeader->m_ucNumber);
				bErrorFound = true;
			}

			if((iTrackIndex >> 1) != poSectorHeader->m_ucTrack)
			{
				printf
				(
					"\n    Sector %2d: bad cylinder number (%d instead of %d)",
					iSector + 1,
					poSectorHeader->m_ucTrack,
					(iTrackIndex >> 1)
				);
				bErrorFound = true;
			}

			if((iTrackIndex & 1) != poSectorHeader->m_ucSide)
			{
				printf
				(
					"\n    Sector %2d: bad side number (%d instead of %d)",
					iSector + 1,
					poSectorHeader->m_ucSide,
					(iTrackIndex & 1)
				);
				bErrorFound = true;
			}

			if(usIDAMCRC != usComputedIDAMCRC)
			{
				printf
				(
					"\n    Sector %2d: bad IDAM CRC (%04x instead of %04x)",
					iSector + 1,
					usIDAMCRC,
					usComputedIDAMCRC
				);
				bErrorFound = true;
			}

			if(usDataCRC != usComputedDataCRC)
			{
				printf
				(
					"\n    Sector %2d: bad data CRC (%04x instead of %04x)",
					iSector + 1,
					usDataCRC,
					usComputedDataCRC
				);
				bErrorFound = true;
			}
		}

		if(!bErrorFound)
		{
			printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b              \b\b\b\b\b\b\b\b\b\b\b\b\b\b");
		}
		else
		{
			printf("\n\n");
		}

		poTracks[iTrackIndex].m_ucSectorsCount = iSector;

		if(poTracks[iTrackIndex].m_ucSectorsCount != 9)
		{
			printf("    Unusual sector count for track (%d instead of 9)\n", poTracks[iTrackIndex].m_ucSectorsCount);
		}
	}

	fclose(g_poFile);

	printf("\nWriting SCP file...\n");

	g_poFile = fopen(argv[2], "w+b");
	if(!g_poFile) vError("Unable to open %s", argv[2]);

	memcpy(oSCPHeader.sig, "SCP", sizeof(oSCPHeader.sig));
	memset(oSCPHeader.tracks_offsets, 0, sizeof(oSCPHeader.tracks_offsets));
	oSCPHeader.version = 0x15;

	oSCPHeader.disk_type = 0x44;
	oSCPHeader.nr_revolutions = 1;
	oSCPHeader.start_track = 0;
	oSCPHeader.end_track = ucNumberOfTracks - 1;
	oSCPHeader.flags = 11;
	oSCPHeader.version = 0x19;

	oSCPHeader.cell_width = 0;
	oSCPHeader.reserved = 0;
	oSCPHeader.checksum = 0;

	fseek(g_poFile, sizeof(oSCPHeader), SEEK_SET);

	for(iTrackIndex = 0; iTrackIndex < ucNumberOfTracks; iTrackIndex++)
	{
		/*~~~~~~~~~~~~~~~*/
		int iByteIndex;
		int iNextMarkGuess;
		int iIAMsFound;
		int iIDAMsFound;
		int iDAMsFound;
		/*~~~~~~~~~~~~~~~*/

		g_uiClock = 0;
		g_uiTotalClock = 0;
		g_bPreviousBit = false;
		g_uiSamplesWritten = 0;

		iSector = 0;
		iNextMarkGuess = 0;

		oSCPHeader.tracks_offsets[iTrackIndex] = ftell(g_poFile);
		fseek(g_poFile, sizeof(oSCPTrackHeader), SEEK_CUR);

		iIAMsFound = 0;
		iIDAMsFound = 0;
		iDAMsFound = 0;

		for
		(
			iByteIndex = 0;
			iByteIndex < (signed) oDMKHeader.m_usTrackLength - (signed) sizeof(poTracks[iTrackIndex].m_ausIDAMOffsets);
			iByteIndex++
		)
		{
			if
			(
				(iSector == 0)
			&&	(poTracks[iTrackIndex].m_pucData[iByteIndex + 1] == 0xC2)
			&&	(poTracks[iTrackIndex].m_pucData[iByteIndex + 2] == 0xC2)
			&&	(poTracks[iTrackIndex].m_pucData[iByteIndex + 3] != 0xC2)
			)
			{
				iIAMsFound++;

				vWriteBits(0x5224, 16, false);
				vWriteBits(0x5224, 16, false);
				vWriteBits(0x5224, 16, false);

				iByteIndex += 2;
			}
			else
			{
				if(!iNextMarkGuess && (iByteIndex == poTracks[iTrackIndex].m_ausIDAMOffsets[iSector] - 3))
				{
					iIDAMsFound++;

					iNextMarkGuess = iByteIndex + 32;
					iSector++;

					vWriteBits(0x4489, 16, false);
					vWriteBits(0x4489, 16, false);
					vWriteBits(0x4489, 16, false);

					iByteIndex += 2;
				}
				else
				{
					if
					(
						iNextMarkGuess
					&&	(iByteIndex > iNextMarkGuess)
					&&	(poTracks[iTrackIndex].m_pucData[iByteIndex + 1] == 0xA1)
					&&	(poTracks[iTrackIndex].m_pucData[iByteIndex + 2] == 0xA1)
					&&	(poTracks[iTrackIndex].m_pucData[iByteIndex + 3] != 0xA1)
					)
					{
						iDAMsFound++;

						iNextMarkGuess = 0;

						vWriteBits(0x4489, 16, false);
						vWriteBits(0x4489, 16, false);
						vWriteBits(0x4489, 16, false);

						iByteIndex += 2;
					}
					else
					{
						vWriteBits(poTracks[iTrackIndex].m_pucData[iByteIndex], 8, true);
					}
				}
			}
		}

		if((iIAMsFound != 1) || (iIDAMsFound != 9) || (iDAMsFound != 9))
		{
			printf("\n  Track %4d: unusual layout (IAMs: %d IDAMs: %d  DAMs: %d)\n", iTrackIndex, iIAMsFound, iIDAMsFound, iDAMsFound);
		}

		memcpy(oSCPTrackHeader.sig, "TRK", sizeof(oSCPHeader.sig));
		oSCPTrackHeader.offset = 0x10;
		oSCPTrackHeader.tracknr = iTrackIndex;
		oSCPTrackHeader.duration = g_uiTotalClock;
		oSCPTrackHeader.nr_samples = g_uiSamplesWritten;

		fseek(g_poFile, oSCPHeader.tracks_offsets[iTrackIndex], SEEK_SET);
		fwrite(&oSCPTrackHeader, sizeof(oSCPTrackHeader), 1, g_poFile);
		fseek(g_poFile, 0, SEEK_END);

		free(poTracks[iTrackIndex].m_pucData);
	}

	free(poTracks);

	fseek(g_poFile, 0, SEEK_SET);
	fwrite(&oSCPHeader, sizeof(oSCPHeader), 1, g_poFile);

	fseek(g_poFile, 16, SEEK_SET);
	oSCPHeader.checksum = 0;

	while((iReadCount = fread(&aucBuffer, 1, sizeof(aucBuffer), g_poFile)) > 0)
	{
		/*~~~~~~~*/
		int iIndex;
		/*~~~~~~~*/

		for(iIndex = 0; iIndex < iReadCount; iIndex++) oSCPHeader.checksum += aucBuffer[iIndex];
	}

	fseek(g_poFile, 0, SEEK_SET);
	fwrite(&oSCPHeader, 16, 1, g_poFile);

	fclose(g_poFile);

	printf("\nDone!\n");
}
